<?php
// Monolith CRUD endpoint for all entities
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/utilities.php';
require_once __DIR__ . '/csrf.php';

// CSRF check for state-changing methods
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
if ($method !== 'GET') {
    $hdrToken = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? '';
    if (!csrf_verify($hdrToken)) {
        respond(['error' => 'CSRF token invalid'], 403);
    }
}

// Helper input readers
function read_int($arr, $key, $default = null) {
    if (!isset($arr[$key])) return $default;
    return (int)$arr[$key];
}
function read_string($arr, $key, $default = null) {
    if (!isset($arr[$key])) return $default;
    return trim((string)$arr[$key]);
}
function read_per_page($arr, $key, $default = 10) {
    $allowed = [10,25,50,100];
    $v = isset($arr[$key]) ? (int)$arr[$key] : $default;
    return in_array($v, $allowed, true) ? $v : $default;
}

// Central validation
function validate_positive_int($value) {
    if (is_numeric($value)) {
        $iv = (int)$value;
        return $iv > 0 ? $iv : null;
    }
    return null;
}
function validate_string_max($value, $maxLen) {
    $s = trim((string)$value);
    if ($s === '') return null;
    if (mb_strlen($s) > $maxLen) return null;
    return $s;
}
function validate_enum($value, $allowed) {
    return in_array($value, $allowed, true) ? $value : null;
}

$ALLOWED_TICKET_STATUS = ['open','in progress','done'];

$entity = isset($_REQUEST['entity']) ? $_REQUEST['entity'] : null;

// Server-side actions (not tied to an entity) --------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_REQUEST['action']) && $_REQUEST['action'] === 'apply_template') {
    // Accept JSON body or form fields
    $raw = file_get_contents('php://input');
    $body = [];
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
    if (stripos($contentType, 'application/json') !== false) {
        $json = json_decode($raw, true);
        if (is_array($json)) $body = $json;
    }
    // fallback to $_POST/REQUEST
    $templateId = isset($body['template_id']) ? validate_positive_int($body['template_id']) : (isset($_REQUEST['template_id']) ? validate_positive_int($_REQUEST['template_id']) : null);
    $ticketId = isset($body['ticket_id']) ? validate_positive_int($body['ticket_id']) : (isset($_REQUEST['ticket_id']) ? validate_positive_int($_REQUEST['ticket_id']) : null);
    if (!$templateId || !$ticketId) respond(['error' => 'template_id and ticket_id required'], 400);

    try {
        // load all template items for the template
        $stmt = $pdo->prepare('SELECT * FROM todo_template_items WHERE template_id = ?');
        $stmt->execute([$templateId]);
        $rows = $stmt->fetchAll();

        // build tree map id -> node
        $map = [];
        foreach ($rows as $r) {
            $r['children'] = [];
            $map[$r['id']] = $r;
        }
        $roots = [];
        foreach ($map as $id => $node) {
            $pid = $node['parent_id'];
            if ($pid && isset($map[$pid])) {
                $map[$pid]['children'][] = &$map[$id];
            } else {
                $roots[] = &$map[$id];
            }
        }

        // recursive insert into todo_items preserving hierarchy
        $insertStmt = $pdo->prepare('INSERT INTO todo_items (ticket_id, parent_id, description, is_completed, actual_minutes) VALUES (?, ?, ?, 0, NULL)');
        $idMap = []; // template_item_id -> new todo_item_id
        $created = 0;

        $pdo->beginTransaction();
        $fn = function($nodes, $parentNewId = null) use (&$fn, &$insertStmt, &$idMap, &$created, $ticketId) {
            foreach ($nodes as $n) {
                $desc = $n['description'] ?? $n['text'] ?? '';
                $desc = trim((string)$desc);
                $pidParam = $parentNewId === null ? null : $parentNewId;
                $insertStmt->execute([$ticketId, $pidParam, $desc]);
                $newId = $insertStmt->rowCount() ? $GLOBALS['pdo']->lastInsertId() : null;
                // As $pdo is not in function scope, fetch lastInsertId via $insertStmt->getConnection? Simpler: use global pdo
                $newId = $GLOBALS['pdo']->lastInsertId();
                if ($newId) $created++;
                // map original id to new id for possible future use
                if (isset($n['id'])) $idMap[$n['id']] = $newId;
                if (!empty($n['children'])) $fn($n['children'], $newId);
            }
        };
        $fn($roots, null);
        $pdo->commit();
        respond(['success' => true, 'created' => $created, 'id_map' => $idMap]);
    } catch (PDOException $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        respond(['error' => $e->getMessage()], 500);
    }
}


switch ($entity) {
    case 'admins':
        // CREATE
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_REQUEST['email'], $_REQUEST['password'])) {
            $email = filter_var($_REQUEST['email'], FILTER_VALIDATE_EMAIL);
            $pwd = (string)$_REQUEST['password'];
            if (!$email) {
                respond(['error' => 'Invalid email'], 400);
            }
            if (mb_strlen($pwd) < 8) {
                respond(['error' => 'Password must be at least 8 characters'], 400);
            }
            $password = password_hash($pwd, PASSWORD_DEFAULT);
            try {
                $stmt = $pdo->prepare('INSERT INTO admins (email, password_hash) VALUES (?, ?)');
                $stmt->execute([$email, $password]);
                respond(['success' => true, 'id' => $pdo->lastInsertId()]);
            } catch (PDOException $e) {
                respond(['error' => $e->getMessage()], 400);
            }
        }
        // READ
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            if (isset($_REQUEST['id'])) {
                $id = validate_positive_int($_REQUEST['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $stmt = $pdo->prepare('SELECT id, email, created_at FROM admins WHERE id = ?');
                $stmt->execute([$id]);
                $admin = $stmt->fetch();
                respond($admin ?: ['error' => 'Not found'], $admin ? 200 : 404);
            } else {
                $stmt = $pdo->query('SELECT id, email, created_at FROM admins');
                respond($stmt->fetchAll());
            }
        }
        // UPDATE
        if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
            parse_str(file_get_contents('php://input'), $_PUT);
            // support JSON bodies (fetch() sends application/json)
            $raw = file_get_contents('php://input');
            $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
            if (stripos($contentType, 'application/json') !== false) {
                $json = json_decode($raw, true);
                if (is_array($json)) {
                    // merge JSON body into _PUT (JSON takes precedence)
                    $_PUT = array_merge($_PUT, $json);
                }
            }

            // Bulk reorder support: { reorder: [ {id:..., position:...}, ... ] } or { positions: { id: position, ... } }
            if (isset($_PUT['reorder']) && is_array($_PUT['reorder'])) {
                $reorder = $_PUT['reorder'];
                try {
                    $pdo->beginTransaction();
                    $stmt = $pdo->prepare('UPDATE todo_items SET position = ? WHERE id = ?');
                    foreach ($reorder as $entry) {
                        if (!is_array($entry)) { $pdo->rollBack(); respond(['error' => 'Invalid reorder entry'], 400); }
                        $id = validate_positive_int($entry['id'] ?? null);
                        $pos = isset($entry['position']) ? (int)$entry['position'] : null;
                        if (!$id || $pos === null || $pos < 0) { $pdo->rollBack(); respond(['error' => 'Invalid reorder entry values'], 400); }
                        $stmt->execute([$pos, $id]);
                    }
                    $pdo->commit();
                    respond(['success' => true]);
                } catch (PDOException $e) {
                    if ($pdo->inTransaction()) $pdo->rollBack();
                    respond(['error' => $e->getMessage()], 500);
                }
            }

            if (isset($_PUT['id'])) {
                $id = validate_positive_int($_PUT['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $fields = [];
                $params = [];
                if (isset($_PUT['email'])) {
                    $email = filter_var($_PUT['email'], FILTER_VALIDATE_EMAIL);
                    if (!$email) respond(['error' => 'Invalid email'], 400);
                    $fields[] = 'email = ?';
                    $params[] = $email;
                }
                if (isset($_PUT['password'])) {
                    $pwd = (string)$_PUT['password'];
                    if (mb_strlen($pwd) < 8) respond(['error' => 'Password must be at least 8 characters'], 400);
                    $fields[] = 'password_hash = ?';
                    $params[] = password_hash($pwd, PASSWORD_DEFAULT);
                }
                if ($fields) {
                    $params[] = $id;
                    $sql = 'UPDATE admins SET ' . implode(', ', $fields) . ' WHERE id = ?';
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute($params);
                    respond(['success' => true]);
                } else {
                    respond(['error' => 'No fields to update'], 400);
                }
            } else {
                respond(['error' => 'Missing id'], 400);
            }
        }
        // DELETE
        if ($_SERVER['REQUEST_METHOD'] === 'DELETE' && isset($_REQUEST['id'])) {
            $id = validate_positive_int($_REQUEST['id']);
            if (!$id) respond(['error' => 'Invalid id'], 400);
            $stmt = $pdo->prepare('DELETE FROM admins WHERE id = ?');
            $stmt->execute([$id]);
            respond(['success' => true]);
        }
        break;
    case 'tickets':
        // CREATE
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && (isset($_REQUEST['title'], $_REQUEST['description'], $_REQUEST['status']) && (isset($_REQUEST['client_id']) || isset($_REQUEST['user_id'])))) {
            $title = validate_string_max($_REQUEST['title'], 100);
            $description = validate_string_max($_REQUEST['description'], 65535); // TEXT
            $status = validate_enum($_REQUEST['status'], $ALLOWED_TICKET_STATUS);
            $clientId = isset($_REQUEST['client_id']) ? validate_positive_int($_REQUEST['client_id']) : null;
            if (!$clientId && isset($_REQUEST['user_id'])) { // backward-compat: map user_id -> client_id
                $clientId = validate_positive_int($_REQUEST['user_id']);
            }
            if (!$title || !$description || !$status || !$clientId) {
                respond(['error' => 'Invalid ticket fields'], 400);
            }
            try {
                $stmt = $pdo->prepare('INSERT INTO tickets (client_id, title, description, status) VALUES (?, ?, ?, ?)');
                $stmt->execute([$clientId, $title, $description, $status]);
                respond(['success' => true, 'id' => $pdo->lastInsertId()]);
            } catch (PDOException $e) {
                respond(['error' => $e->getMessage()], 400);
            }
        }
        // READ
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            if (isset($_REQUEST['id'])) {
                $id = validate_positive_int($_REQUEST['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $stmt = $pdo->prepare('SELECT * FROM tickets WHERE id = ?');
                $stmt->execute([$id]);
                $ticket = $stmt->fetch();
                respond($ticket ?: ['error' => 'Not found'], $ticket ? 200 : 404);
            } else {
                // Server-side filtering and pagination when any related param is provided
                $hasPagingOrFilter = isset($_GET['page']) || isset($_GET['per_page']) || isset($_GET['q']) || isset($_GET['status']) || isset($_GET['client_id']) || isset($_REQUEST['page']) || isset($_REQUEST['per_page']);
                if ($hasPagingOrFilter) {
                    $page = max(1, read_int($_GET, 'page', 1));
                    $perPage = read_per_page($_GET, 'per_page', 10);
                    $offset = ($page - 1) * $perPage;
                    $where = [];
                    $params = [];

                    $status = read_string($_GET, 'status');
                    if ($status !== null && $status !== '') {
                        $statusV = validate_enum($status, $ALLOWED_TICKET_STATUS);
                        if (!$statusV) respond(['error' => 'Invalid status'], 400);
                        $where[] = 'status = ?';
                        $params[] = $statusV;
                    }
                    $clientId = read_int($_GET, 'client_id');
                    if ($clientId) {
                        $cid = validate_positive_int($clientId);
                        if (!$cid) respond(['error' => 'Invalid client_id'], 400);
                        $where[] = 'client_id = ?';
                        $params[] = $cid;
                    }
                    $q = read_string($_GET, 'q');
                    if ($q !== null && $q !== '') {
                        // limit q length to avoid abuse
                        if (mb_strlen($q) > 200) respond(['error' => 'Search too long'], 400);
                        $where[] = '(title LIKE ? OR description LIKE ?)';
                        $like = '%' . $q . '%';
                        $params[] = $like;
                        $params[] = $like;
                    }

                    $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

                    // Count
                    $countSql = 'SELECT COUNT(*) FROM tickets ' . $whereSql;
                    $countStmt = $pdo->prepare($countSql);
                    $countStmt->execute($params);
                    $total = (int)$countStmt->fetchColumn();
                    $totalPages = max(1, (int)ceil($total / $perPage));
                    if ($page > $totalPages) { $page = $totalPages; $offset = ($page - 1) * $perPage; }

                    // Page data
                    $dataSql = 'SELECT * FROM tickets ' . $whereSql . ' ORDER BY created_at DESC LIMIT :limit OFFSET :offset';
                    $dataStmt = $pdo->prepare($dataSql);
                    $i = 1;
                    foreach ($params as $p) {
                        $dataStmt->bindValue($i, $p, is_int($p) ? PDO::PARAM_INT : PDO::PARAM_STR);
                        $i++;
                    }
                    $dataStmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
                    $dataStmt->bindValue(':offset', $offset, PDO::PARAM_INT);
                    $dataStmt->execute();
                    $rows = $dataStmt->fetchAll();

                    respond([
                        'meta' => [
                            'page' => $page,
                            'per_page' => $perPage,
                            'total' => $total,
                            'total_pages' => $totalPages
                        ],
                        'data' => $rows
                    ]);
                } else {
                    // Legacy: return all
                    $stmt = $pdo->query('SELECT * FROM tickets ORDER BY created_at DESC');
                    respond($stmt->fetchAll());
                }
            }
        }
        // UPDATE
        if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
            parse_str(file_get_contents('php://input'), $_PUT);
            if (isset($_PUT['id'])) {
                $id = validate_positive_int($_PUT['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $fields = [];
                $params = [];
                if (isset($_PUT['title'])) {
                    $title = validate_string_max($_PUT['title'], 100);
                    if (!$title) respond(['error' => 'Invalid title'], 400);
                    $fields[] = 'title = ?';
                    $params[] = $title;
                }
                if (isset($_PUT['description'])) {
                    $desc = validate_string_max($_PUT['description'], 65535);
                    if (!$desc) respond(['error' => 'Invalid description'], 400);
                    $fields[] = 'description = ?';
                    $params[] = $desc;
                }
                if (isset($_PUT['status'])) {
                    $st = validate_enum($_PUT['status'], $ALLOWED_TICKET_STATUS);
                    if (!$st) respond(['error' => 'Invalid status'], 400);
                    $fields[] = 'status = ?';
                    $params[] = $st;
                }
                // normalize client_id
                if (isset($_PUT['client_id']) || isset($_PUT['user_id'])) {
                    $cid = isset($_PUT['client_id']) ? $_PUT['client_id'] : $_PUT['user_id'];
                    $cid = validate_positive_int($cid);
                    if (!$cid) respond(['error' => 'Invalid client_id'], 400);
                    $fields[] = 'client_id = ?';
                    $params[] = $cid;
                }
                if ($fields) {
                    $params[] = $id;
                    $sql = 'UPDATE tickets SET ' . implode(', ', $fields) . ' WHERE id = ?';
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute($params);
                    respond(['success' => true]);
                } else {
                    respond(['error' => 'No fields to update'], 400);
                }
            } else {
                respond(['error' => 'Missing id'], 400);
            }
        }
        // DELETE
        if ($_SERVER['REQUEST_METHOD'] === 'DELETE' && isset($_REQUEST['id'])) {
            $id = validate_positive_int($_REQUEST['id']);
            if (!$id) respond(['error' => 'Invalid id'], 400);
            $stmt = $pdo->prepare('DELETE FROM tickets WHERE id = ?');
            $stmt->execute([$id]);
            respond(['success' => true]);
        }
        break;
    case 'todo_items':
        // CREATE
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_REQUEST['ticket_id'], $_REQUEST['description'])) {
            $ticket_id = validate_positive_int($_REQUEST['ticket_id']);
            $description = validate_string_max($_REQUEST['description'], 65535);
            $parent_id = isset($_REQUEST['parent_id']) && $_REQUEST['parent_id'] !== '' ? validate_positive_int($_REQUEST['parent_id']) : null;
            $is_completed = isset($_REQUEST['is_completed']) ? (int)!!$_REQUEST['is_completed'] : 0;
            $actual_minutes = isset($_REQUEST['actual_minutes']) ? (int)$_REQUEST['actual_minutes'] : null;
            if (!$ticket_id || !$description) respond(['error' => 'Invalid todo item fields'], 400);
            try {
                $stmt = $pdo->prepare('INSERT INTO todo_items (ticket_id, parent_id, description, is_completed, actual_minutes) VALUES (?, ?, ?, ?, ?)');
                $stmt->execute([$ticket_id, $parent_id, $description, $is_completed, $actual_minutes]);
                respond(['success' => true, 'id' => $pdo->lastInsertId()]);
            } catch (PDOException $e) {
                respond(['error' => $e->getMessage()], 400);
            }
        }
        // READ
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            if (isset($_REQUEST['id'])) {
                $id = validate_positive_int($_REQUEST['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $stmt = $pdo->prepare('SELECT * FROM todo_items WHERE id = ?');
                $stmt->execute([$id]);
                $item = $stmt->fetch();
                respond($item ?: ['error' => 'Not found'], $item ? 200 : 404);
            } else if (isset($_REQUEST['ticket_id'])) {
                $tid = validate_positive_int($_REQUEST['ticket_id']);
                if (!$tid) respond(['error' => 'Invalid ticket_id'], 400);
                $stmt = $pdo->prepare('SELECT * FROM todo_items WHERE ticket_id = ?');
                $stmt->execute([$tid]);
                respond($stmt->fetchAll());
            } else {
                $stmt = $pdo->query('SELECT * FROM todo_items');
                respond($stmt->fetchAll());
            }
        }
        // UPDATE
        if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
            parse_str(file_get_contents('php://input'), $_PUT);
            if (isset($_PUT['id'])) {
                $id = validate_positive_int($_PUT['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $fields = [];
                $params = [];
                if (isset($_PUT['ticket_id'])) {
                    $tid = validate_positive_int($_PUT['ticket_id']);
                    if (!$tid) respond(['error' => 'Invalid ticket_id'], 400);
                    $fields[] = 'ticket_id = ?';
                    $params[] = $tid;
                }
                if (isset($_PUT['parent_id'])) {
                    $pid = $_PUT['parent_id'] === '' ? null : validate_positive_int($_PUT['parent_id']);
                    if ($_PUT['parent_id'] !== '' && !$pid) respond(['error' => 'Invalid parent_id'], 400);
                    $fields[] = 'parent_id = ?';
                    $params[] = $pid;
                }
                if (isset($_PUT['description'])) {
                    $desc = validate_string_max($_PUT['description'], 65535);
                    if (!$desc) respond(['error' => 'Invalid description'], 400);
                    $fields[] = 'description = ?';
                    $params[] = $desc;
                }
                if (isset($_PUT['is_completed'])) {
                    $fields[] = 'is_completed = ?';
                    $params[] = (int)!!$_PUT['is_completed'];
                }
                if (isset($_PUT['actual_minutes'])) {
                    $fields[] = 'actual_minutes = ?';
                    $params[] = (int)$_PUT['actual_minutes'];
                }
                if ($fields) {
                    $params[] = $id;
                    $sql = 'UPDATE todo_items SET ' . implode(', ', $fields) . ' WHERE id = ?';
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute($params);
                    respond(['success' => true]);
                } else {
                    respond(['error' => 'No fields to update'], 400);
                }
            } else {
                respond(['error' => 'Missing id'], 400);
            }
        }
        // DELETE
        if ($_SERVER['REQUEST_METHOD'] === 'DELETE' && isset($_REQUEST['id'])) {
            $id = validate_positive_int($_REQUEST['id']);
            if (!$id) respond(['error' => 'Invalid id'], 400);
            $stmt = $pdo->prepare('DELETE FROM todo_items WHERE id = ?');
            $stmt->execute([$id]);
            respond(['success' => true]);
        }
        break;
    case 'clients':
        // CREATE
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_REQUEST['name'])) {
            $name = validate_string_max($_REQUEST['name'], 100);
            if (!$name) respond(['error' => 'Invalid client name'], 400);
            try {
                $stmt = $pdo->prepare('INSERT INTO clients (name) VALUES (?)');
                $stmt->execute([$name]);
                respond(['success' => true, 'id' => $pdo->lastInsertId()]);
            } catch (PDOException $e) {
                respond(['error' => $e->getMessage()], 400);
            }
        }
        // READ
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            if (isset($_REQUEST['id'])) {
                $id = validate_positive_int($_REQUEST['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $stmt = $pdo->prepare('SELECT * FROM clients WHERE id = ?');
                $stmt->execute([$id]);
                $client = $stmt->fetch();
                respond($client ?: ['error' => 'Not found'], $client ? 200 : 404);
            } else {
                $hasPagingOrFilter = isset($_GET['page']) || isset($_GET['per_page']) || isset($_GET['q']) || isset($_REQUEST['page']) || isset($_REQUEST['per_page']);
                // show_archived param controls whether inactive clients are returned. default: false (only active)
                $showArchived = isset($_GET['show_archived']) && ($_GET['show_archived'] === '1' || $_GET['show_archived'] === 'true');
                if ($hasPagingOrFilter) {
                    $page = max(1, read_int($_GET, 'page', 1));
                    $perPage = read_per_page($_GET, 'per_page', 10);
                    $offset = ($page - 1) * $perPage;
                    $params = [];
                    $where = [];
                    $q = read_string($_GET, 'q');
                    if ($q !== null && $q !== '') {
                        if (mb_strlen($q) > 200) respond(['error' => 'Search too long'], 400);
                        $where[] = 'name LIKE ?';
                        $params[] = '%' . $q . '%';
                    }
                    if (!$showArchived) {
                        $where[] = 'is_active = 1';
                    }
                    $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

                    $countSql = 'SELECT COUNT(*) FROM clients ' . $whereSql;
                    $countStmt = $pdo->prepare($countSql);
                    $countStmt->execute($params);
                    $total = (int)$countStmt->fetchColumn();
                    $totalPages = max(1, (int)ceil($total / $perPage));
                    if ($page > $totalPages) { $page = $totalPages; $offset = ($page - 1) * $perPage; }

                    $dataSql = 'SELECT * FROM clients ' . $whereSql . ' ORDER BY created_at DESC LIMIT :limit OFFSET :offset';
                    $dataStmt = $pdo->prepare($dataSql);
                    $i = 1;
                    foreach ($params as $p) {
                        $dataStmt->bindValue($i, $p, PDO::PARAM_STR);
                        $i++;
                    }
                    $dataStmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
                    $dataStmt->bindValue(':offset', $offset, PDO::PARAM_INT);
                    $dataStmt->execute();
                    $rows = $dataStmt->fetchAll();

                    respond([
                        'meta' => [
                            'page' => $page,
                            'per_page' => $perPage,
                            'total' => $total,
                            'total_pages' => $totalPages
                        ],
                        'data' => $rows
                    ]);
                } else {
                    $sql = 'SELECT * FROM clients ' . ($showArchived ? '' : 'WHERE is_active = 1') . ' ORDER BY created_at DESC';
                    $stmt = $pdo->query($sql);
                    respond($stmt->fetchAll());
                }
            }
        }
        // UPDATE
        if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
            parse_str(file_get_contents('php://input'), $_PUT);
            if (isset($_PUT['id'])) {
                $id = validate_positive_int($_PUT['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $fields = [];
                $params = [];
                if (isset($_PUT['name'])) {
                    $name = validate_string_max($_PUT['name'], 100);
                    if (!$name) respond(['error' => 'Invalid name'], 400);
                    $fields[] = 'name = ?';
                    $params[] = $name;
                }
                if ($fields) {
                    $params[] = $id;
                    $sql = 'UPDATE clients SET ' . implode(', ', $fields) . ' WHERE id = ?';
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute($params);
                    respond(['success' => true]);
                } else {
                    respond(['error' => 'No fields to update'], 400);
                }
            } else {
                respond(['error' => 'Missing id'], 400);
            }
        }
        // DELETE -> soft-delete (set is_active = 0)
        if ($_SERVER['REQUEST_METHOD'] === 'DELETE' && isset($_REQUEST['id'])) {
            $id = validate_positive_int($_REQUEST['id']);
            if (!$id) respond(['error' => 'Invalid id'], 400);
            $stmt = $pdo->prepare('UPDATE clients SET is_active = 0 WHERE id = ?');
            $stmt->execute([$id]);
            respond(['success' => true]);
        }
        break;
    case 'todo_templates':
        // CREATE
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_REQUEST['client_id'], $_REQUEST['name'])) {
            $client_id = validate_positive_int($_REQUEST['client_id']);
            $name = validate_string_max($_REQUEST['name'], 100);
            $description = isset($_REQUEST['description']) ? validate_string_max($_REQUEST['description'], 65535) : null;
            if (!$client_id || !$name) respond(['error' => 'Invalid template fields'], 400);
            try {
                $stmt = $pdo->prepare('INSERT INTO todo_templates (client_id, name, description) VALUES (?, ?, ?)');
                $stmt->execute([$client_id, $name, $description]);
                respond(['success' => true, 'id' => $pdo->lastInsertId()]);
            } catch (PDOException $e) {
                respond(['error' => $e->getMessage()], 400);
            }
        }
        // READ
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            if (isset($_REQUEST['id'])) {
                $id = validate_positive_int($_REQUEST['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $stmt = $pdo->prepare('SELECT * FROM todo_templates WHERE id = ?');
                $stmt->execute([$id]);
                $template = $stmt->fetch();
                respond($template ?: ['error' => 'Not found'], $template ? 200 : 404);
            } else if (isset($_REQUEST['client_id'])) {
                $cid = validate_positive_int($_REQUEST['client_id']);
                if (!$cid) respond(['error' => 'Invalid client_id'], 400);
                $stmt = $pdo->prepare('SELECT * FROM todo_templates WHERE client_id = ?');
                $stmt->execute([$cid]);
                respond($stmt->fetchAll());
            } else {
                $stmt = $pdo->query('SELECT * FROM todo_templates');
                respond($stmt->fetchAll());
            }
        }
        // UPDATE
        if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
            parse_str(file_get_contents('php://input'), $_PUT);
            if (isset($_PUT['id'])) {
                $id = validate_positive_int($_PUT['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $fields = [];
                $params = [];
                if (isset($_PUT['client_id'])) {
                    $cid = validate_positive_int($_PUT['client_id']);
                    if (!$cid) respond(['error' => 'Invalid client_id'], 400);
                    $fields[] = 'client_id = ?';
                    $params[] = $cid;
                }
                if (isset($_PUT['name'])) {
                    $name = validate_string_max($_PUT['name'], 100);
                    if (!$name) respond(['error' => 'Invalid name'], 400);
                    $fields[] = 'name = ?';
                    $params[] = $name;
                }
                if (isset($_PUT['description'])) {
                    $desc = validate_string_max($_PUT['description'], 65535);
                    if (!$desc) respond(['error' => 'Invalid description'], 400);
                    $fields[] = 'description = ?';
                    $params[] = $desc;
                }
                if ($fields) {
                    $params[] = $id;
                    $sql = 'UPDATE todo_templates SET ' . implode(', ', $fields) . ' WHERE id = ?';
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute($params);
                    respond(['success' => true]);
                } else {
                    respond(['error' => 'No fields to update'], 400);
                }
            } else {
                respond(['error' => 'Missing id'], 400);
            }
        }
        // DELETE
        if ($_SERVER['REQUEST_METHOD'] === 'DELETE' && isset($_REQUEST['id'])) {
            $id = validate_positive_int($_REQUEST['id']);
            if (!$id) respond(['error' => 'Invalid id'], 400);
            $stmt = $pdo->prepare('DELETE FROM todo_templates WHERE id = ?');
            $stmt->execute([$id]);
            respond(['success' => true]);
        }
        break;
    case 'todo_template_items':
        // CREATE
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_REQUEST['template_id'], $_REQUEST['description'])) {
            $template_id = validate_positive_int($_REQUEST['template_id']);
            $description = validate_string_max($_REQUEST['description'], 65535);
            $parent_id = isset($_REQUEST['parent_id']) && $_REQUEST['parent_id'] !== '' ? validate_positive_int($_REQUEST['parent_id']) : null;
            $estimated_minutes = isset($_REQUEST['estimated_minutes']) ? (int)$_REQUEST['estimated_minutes'] : null;
            if (!$template_id || !$description) respond(['error' => 'Invalid template item fields'], 400);
            try {
                $stmt = $pdo->prepare('INSERT INTO todo_template_items (template_id, parent_id, description, estimated_minutes) VALUES (?, ?, ?, ?)');
                $stmt->execute([$template_id, $parent_id, $description, $estimated_minutes]);
                respond(['success' => true, 'id' => $pdo->lastInsertId()]);
            } catch (PDOException $e) {
                respond(['error' => $e->getMessage()], 400);
            }
        }
        // READ
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            if (isset($_REQUEST['id'])) {
                $id = validate_positive_int($_REQUEST['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $stmt = $pdo->prepare('SELECT * FROM todo_template_items WHERE id = ?');
                $stmt->execute([$id]);
                $item = $stmt->fetch();
                respond($item ?: ['error' => 'Not found'], $item ? 200 : 404);
            } else if (isset($_REQUEST['template_id'])) {
                $tid = validate_positive_int($_REQUEST['template_id']);
                if (!$tid) respond(['error' => 'Invalid template_id'], 400);
                $stmt = $pdo->prepare('SELECT * FROM todo_template_items WHERE template_id = ?');
                $stmt->execute([$tid]);
                respond($stmt->fetchAll());
            } else {
                $stmt = $pdo->query('SELECT * FROM todo_template_items');
                respond($stmt->fetchAll());
            }
        }
        // UPDATE
        if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
            parse_str(file_get_contents('php://input'), $_PUT);
            if (isset($_PUT['id'])) {
                $id = validate_positive_int($_PUT['id']);
                if (!$id) respond(['error' => 'Invalid id'], 400);
                $fields = [];
                $params = [];
                if (isset($_PUT['template_id'])) {
                    $tid = validate_positive_int($_PUT['template_id']);
                    if (!$tid) respond(['error' => 'Invalid template_id'], 400);
                    $fields[] = 'template_id = ?';
                    $params[] = $tid;
                }
                if (isset($_PUT['parent_id'])) {
                    $pid = $_PUT['parent_id'] === '' ? null : validate_positive_int($_PUT['parent_id']);
                    if ($_PUT['parent_id'] !== '' && !$pid) respond(['error' => 'Invalid parent_id'], 400);
                    $fields[] = 'parent_id = ?';
                    $params[] = $pid;
                }
                if (isset($_PUT['description'])) {
                    $desc = validate_string_max($_PUT['description'], 65535);
                    if (!$desc) respond(['error' => 'Invalid description'], 400);
                    $fields[] = 'description = ?';
                    $params[] = $desc;
                }
                if (isset($_PUT['estimated_minutes'])) {
                    $fields[] = 'estimated_minutes = ?';
                    $params[] = (int)$_PUT['estimated_minutes'];
                }
                if ($fields) {
                    $params[] = $id;
                    $sql = 'UPDATE todo_template_items SET ' . implode(', ', $fields) . ' WHERE id = ?';
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute($params);
                    respond(['success' => true]);
                } else {
                    respond(['error' => 'No fields to update'], 400);
                }
            } else {
                respond(['error' => 'Missing id'], 400);
            }
        }
        // DELETE
        if ($_SERVER['REQUEST_METHOD'] === 'DELETE' && isset($_REQUEST['id'])) {
            $id = validate_positive_int($_REQUEST['id']);
            if (!$id) respond(['error' => 'Invalid id'], 400);
            $stmt = $pdo->prepare('DELETE FROM todo_template_items WHERE id = ?');
            $stmt->execute([$id]);
            respond(['success' => true]);
        }
        break;
    default:
        respond(['error' => 'Invalid entity'], 400);
}
