# Dashboard Test Checklist (admin/dashboard.php)

This file lists manual and automated test cases to validate `admin/dashboard.php`.

## Scope / Contract
- Inputs: authenticated admin session; client-side fetches to `/includes/crud.php` for `tickets` and `clients`.
- Expected response shape for GET: `{ meta: { page, per_page, total, total_pages }, data: [...] }`.
- Mutations (POST/PUT/DELETE) must include `X-CSRF-Token` header; success/error must be reflected to the UI via toasts and refetches.

## Smoke / Happy-path Manual Tests
1. Page load
   - Preconditions: valid admin session.
   - Steps: Open `/admin/dashboard.php`.
   - Expected:
     - `<meta name="csrf-token">` exists and non-empty.
     - Tickets and Clients sections show loading, then render rows and pager info.
     - Prev/Next controls reflect page bounds.

2. Pagination navigation
   - Steps: Click Next/Prev on Tickets and Clients.
   - Expected: loading indicator visible, controls disabled while loading, table updates and pager info changes.

3. Per-page change
   - Steps: Change per-page selects for Tickets and Clients.
   - Expected: page requests with new per_page; UI updates accordingly.

4. Search and filter
   - Steps: Use ticket search and status filter.
   - Expected: current page rows filtered client-side; server-side `q`/`status` parameters used when fetching pages.

5. Create Ticket
   - Steps: Add Ticket (fill and submit form).
   - Expected: optimistic row appears, loading/disable applied, on success toast and table refresh; on failure toast and temp row removed.

6. Edit Ticket
   - Steps: Edit a ticket inline and save.
   - Expected: optimistic UI update, loading applied; on success toast and refetch; on failure toast and refetch/revert.

7. Change ticket status
   - Steps: Click status buttons (Open/In Progress/Done).
   - Expected: optimistic update, on success toast, on failure revert and toast.

8. Delete Ticket
   - Steps: Delete a ticket and confirm.
   - Expected: optimistic removal, on success toast, on failure refetch and restore.

9. Create/Edit/Delete Client
   - Steps: Same as tickets but for Clients.
   - Expected: analogous behavior with clients section loading and disables.

## Error & Edge Cases
1. Server 500 for GET
   - Expected: toast error, loading hides, controls re-enabled, table unchanged.

2. Invalid JSON from API
   - Expected: parse error handled gracefully, toast shown.

3. Missing/invalid CSRF on mutation
   - Steps: remove/alter meta tag and attempt mutation.
   - Expected: 403 JSON error from server, toast shown, optimistic UI reverted.

4. Double-submit prevention
   - Steps: rapidly click create multiple times.
   - Expected: UI prevents duplicate requests via control disabling.

5. Empty state
   - Steps: database contains no tickets/clients.
   - Expected: tables empty, pager shows "Page 1 / 1 — 0 items".

6. Large result sets / performance
   - Steps: populate many rows, change per_page to high value.
   - Expected: acceptable render time and correct paging info.

7. Unauthenticated access
   - Steps: Visit page without session.
   - Expected: redirect to login or 401 handled by server.

8. XSS test
   - Steps: create tickets/clients with HTML/script-like content.
   - Expected: UI escapes content, no script execution.

## JS / Component Tests (unit / integration)
- apiFetch
  - Verify header `X-CSRF-Token` present for non-GET requests.
  - Verify `opts.section` toggles loading and disables controls during fetch.
  - Simulate non-OK responses: ensures toast is shown and apiFetch throws.

- renderTicketsFromApi / renderClientsFromApi
  - Feed mocked JSON with data and meta; assert DOM rows and pager info set correctly.

- filterTickets / filterClients
  - Ensure client-side filtering hides non-matching rows (case-insensitive).

- escapeHtml / escapeHtmlAttr
  - Test with special characters, quotes, tags; ensure result is safe HTML/text.

## Security / Contract Tests
- CSRF presence: verify meta token value equals server session token.
- CSRF enforcement: POST to `/admin/ajax_add_item.php` and other ajax endpoints without header returns 403 JSON.
- CRUD API contract: GET returns `meta` and `data`, mutations respect status codes and message shapes.

## Accessibility / UX Checks
- Keyboard navigation: Tab/Shift+Tab reaches Add/Edit/Delete, Enter activates.
- Toasts: appear and time out; consider adding ARIA attributes (aria-live) to container.
- Controls: disabled state visible (opacity change) and uses `disabled` attribute for semantics.

## Automation suggestions (priority)
1. Playwright end-to-end for core happy path (login -> add client -> add ticket -> edit -> delete).
2. Unit tests for apiFetch and rendering functions (mock fetch). Use node + Jest or in-browser test harness.
3. Small PHP integration script to call `/includes/crud.php` with sample params and validate JSON schema.

## Test data samples
- Ticket: title="Test ticket <script>", description with unicode and HTML-like strings, status="open", client_id=existing id.
- Client: name with quotes, unicode, long strings.

## Quick manual test script (copy/paste)
1. Login as admin.
2. Open `/admin/dashboard.php`.
3. Add a client.
4. Add a ticket for that client.
5. Edit the ticket, change status, delete the ticket.
6. Change per-page and navigate pages.
7. Simulate server error (via devtools or staged API) and confirm error handling.

---

If you want, I can now:
- Scaffold a Playwright test file for the happy path and a small Jest test for `apiFetch` (mocking fetch). Tell me which framework you prefer (Playwright or Cypress for E2E; Jest or Mocha for unit tests).
- Or I can expand this checklist into a formal test plan with step-by-step screenshots and expected HTTP request/response samples.

## API Contract: Bulk Reorder (todo_items)

- Endpoint: PUT /includes/crud.php
- Query/body: entity=todo_items (can be in query or JSON body)
- Request JSON shape (application/json):

  {
    "entity": "todo_items",
    "reorder": [
      { "id": 123, "position": 0 },
      { "id": 124, "position": 1 },
      { "id": 125, "position": 2 }
    ]
  }

- Behavior: Server updates `position` for each `id` in a single transaction. Returns 200 { "success": true } on success. On validation error or DB failure returns an error JSON and appropriate status (400/500).

## CRUD API (`includes/crud.php`) Test Checklist
This list targets `includes/crud.php`. Run these as integration tests (API-level) against a test database and test user session.

1. Authentication & CSRF
   - Verify GET requests work without CSRF header.
   - Verify POST/PUT/DELETE require `X-CSRF-Token`. Missing/invalid token returns 403 + { error }.

2. Entity coverage (happy paths)
   - `admins`: create, get (by id), list, update (email/password), delete.
   - `clients`: create, list paging/filter, update, delete.
   - `tickets`: create (with client_id), list with pagination & filters (q/status), get by id, update status/title/description, delete.
   - `todo_items`: create (ticket_id + description), list by ticket_id, get by id, update fields (description, parent_id, is_completed), delete.
   - `todo_templates` and `todo_template_items`: create, list by client/template, get by id, update, delete.

3. Validation & errors
   - Send invalid payloads (missing required fields, invalid ids, oversized strings) to each entity's CREATE/UPDATE endpoints and assert 400 with helpful error message.
   - For tickets: invalid `status` values should return 400.
   - For pagination: negative page/per_page should be sanitized/302/400 as appropriate.

4. Pagination semantics
   - For `tickets` and `clients`: create > per_page items and ensure meta.total and total_pages computed correctly for various per_page values.
   - Page bounds: if requested page > total_pages, API should return last page and correct meta.page.

5. Bulk reorder (new)
   - Create several todo_items for one ticket (capture ids).
   - Call PUT /includes/crud.php with JSON body { entity: 'todo_items', reorder: [ {id, position}, ... ] } changing the positions.
   - Assert response success and then GET items by ticket_id and confirm positions changed accordingly.
   - Negative test: send malformed reorder entries (missing id/position or negative position) and assert 400.
   - DB error test: simulate DB failure (e.g., invalid id) and ensure transaction rolls back and error returned.

6. Concurrency & atomicity
   - Simulate concurrent reorder and update requests and confirm no partial state (transaction ensures atomicity).

7. Security & edge cases
   - SQL injection attempts in string fields should be safely handled (use prepared statements).
   - Ensure foreign key constraints work (e.g., deleting a ticket cascades todo_items if configured).

8. Response content-type and error shapes
   - All responses must be JSON with appropriate HTTP status codes.
   - Error responses include an `error` key with a human-readable message.

9. Backward compatibility
   - Verify legacy behaviors (e.g., `user_id` mapping to `client_id` in tickets) still work.

10. Performance
   - Bulk reorder: measure performance for N items (50, 100, 500) and check timeouts; consider adding a dedicated bulk endpoint if needed.
