# Project Plan (Structured)

Status legend: [x] done | [~] in progress | [ ] todo

## 0) Foundations
- [x] Database schema file present (`migrations/init.sql`)
- [x] DB connection (`includes/db.php`)
- [x] Auth gate for admin routes (`includes/auth.php`)
- [x] Utilities (`includes/utilities.php`)
- [x] Consolidated monolith CRUD endpoint (`includes/crud.php`)
- [x] Prepared statements across all SQL
- [x] Fix schema reference: `tickets.user_id` -> `tickets.client_id` with FK to `clients(id)` (MySQL-compliant DDL, indexes, FKs)

## 1) Authentication & Registration
- [x] Admin login uses `password_verify`, friendly errors, rate limiting (IP + session)
- [x] Registration page with token-based invite flow (`admin/register.php`, `admin_invites` table)
- [x] Login hardening: rate-limit attempts, lockout window, server-side validation
- [x] CSRF tokens enforced for POST/PUT/DELETE (header `X-CSRF-Token`)
- [x] Secure session cookies: HttpOnly, SameSite=Lax, Secure (on HTTPS)
- [x] Admin Invite UI (`admin/create_invite.php`): create, list, revoke invites with CSRF

Prompts remaining:
- “Add minimal admin role model if different privilege levels are needed later.”

## 2) Dashboard
- [x] Auth-protected dashboard (`admin/dashboard.php`)
- [x] Tickets and Clients listing
- [x] Ticket actions via AJAX: add, inline edit, delete, status
- [x] Client actions via AJAX: add, inline edit, delete
- [x] Non-blocking toasts
- [x] Client-side filters/search (tickets by status/text; clients by name)
- [x] Pagination for large lists
- [x] Clipboard button for ticket link

Prompts:
- “Add pagination to dashboard lists (10/25/50 per page) with prev/next and server-side LIMIT/OFFSET.”
- “Add a ‘Copy link’ button per ticket that copies a shareable URL to clipboard.”

## 3) Ticket Management
- [x] CRUD API for tickets via `includes/crud.php` (entity=tickets)
- [x] Dashboard create/edit/delete/status
- [x] Ticket details page to manage todo items for a ticket
- [x] Option to apply a template to a ticket (spawn initial `todo_items` from `todo_template_items`)

Prompts:
- “Create `admin/ticket.php?id={id}` page: list, add, edit, reorder, mark-complete todo items for the ticket with AJAX to `crud.php` (entity=todo_items).”
- “Add ‘Apply Template’ action: choose client template -> create items and subitems under the ticket.”

## 4) Todo Items
- [x] CRUD API `entity=todo_items`
- [x] UI on ticket page for nested subtasks (parent/child), complete/in-progress toggle, inline edit, drag-sort (optional)

Prompt:
- “Build nested todo UI in `admin/ticket.php` with expand/collapse, inline edits, and completion toggles wired to `crud.php`.”

## 5) Client Management
- [x] CRUD API `entity=clients`
- [x] Dashboard add/edit/delete
- [x] Soft-delete (or ‘archive’) clients instead of hard delete

Prompt:
- “Add `clients.is_active BOOLEAN DEFAULT TRUE`; adjust delete to soft-delete; add filter to show/hide archived.”

## 6) Todo Templates
- [x] CRUD APIs `entity=todo_templates`, `entity=todo_template_items`
- [x] UI to create/edit templates and nested items per client
- [x] Apply templates to tickets (see Ticket Management section)

### New: Time estimates & tracking
- [ ] Add `estimated_minutes` (INT NULL) to `todo_template_items` and `actual_minutes` (INT NULL) to `todo_items`.
- [ ] Update migrations (add SQL files) and database schema.
- [ ] Update `includes/crud.php` to accept `estimated_minutes` on template item create/update and `actual_minutes` on todo item create/update.
- [ ] Update `admin/templates.php` UI to allow entering an estimated minutes value per template item.
- [ ] Update `admin/ticket.php` UI to allow entering actual minutes per todo item and show per-ticket average actual minutes and per-template average estimated minutes when applying templates.
- [ ] Add tests and smoke checks for estimate/actual persistence and average computations.

Prompt:
- “Create `admin/templates.php?client_id={id}` for managing templates and nested template items with AJAX.”

## 7) Security
- [x] SQL injection mitigated via prepared statements
- [x] CSRF protection on state-changing requests (includes/crud.php; admin pages)
- [ ] Input validation and sanitization (length, formats)
- [ ] Output encoding in all templates (use `htmlspecialchars` consistently; avoid unsafe innerHTML)
- [ ] Authorization rules (ensure only admins can call `crud.php`; consider role levels)

Prompts:
- “Add centralized input validation helpers (ints, strings with max length, enums for status) and apply in `includes/crud.php`.”
- “Add a simple policy layer to restrict sensitive operations by role if needed.”

## 8) Testing & QA
- [ ] Postman testing for all `includes/crud.php` entities and operations (GET, POST, PUT, DELETE)
- [ ] Happy-path and error-path tests (missing fields, invalid IDs, unauthorized)
- [ ] Basic browser smoke tests on dashboard and ticket page

Prompt:
- “Create Postman collection with environments; add requests for all CRUD combinations; include CSRF header.”

## 9) Deployment & Ops
- [ ] Environment config (.env) for DB creds instead of hardcoded values
- [ ] Error logging/rotation; disable display_errors in production
- [ ] Web server config: deny direct access to non-entry PHP where appropriate

Prompts:
- “Refactor `includes/db.php` to read creds from environment; add `.env.example` and load via `vlucas/phpdotenv`.”
- “Create production `php.ini` overrides (disable display_errors), and a simple error log setup.”

---

Backlog / Nice-to-haves
- [ ] Pagination, sorting, and column filters everywhere
- [ ] Role-based access control (RBAC) if multiple admin roles
- [ ] Activity log (who changed what)
- [ ] Export tickets or clients to CSV

## UI Modals & UX Improvements (new)
These are focused, high-value modal and UX improvements that make template and ticket workflows faster and more accessible. Each entry includes a short developer prompt to implement it.

- Modal: Template Apply modal [priority: high]
	- Purpose: Replace prompt-based apply with a first-class modal that lets admins search/select a ticket, preview template summary (counts, estimated minutes total), show progress while applying, and confirm or cancel.
	- Implementation notes:
		- Modal with ticket search/autocomplete (calls `/includes/crud.php?entity=tickets&q=...&client_id=...`).
		- Template summary: top-level item count, total estimated minutes (sum of `estimated_minutes`), and a small tree preview.
		- Apply action calls a new server endpoint (recommended) or performs the recursive client-side create while showing progress.
	- Prompt for developer: "Create accessible modal that fetches matching tickets, supports keyboard navigation, shows template summary (items and total estimated minutes), and displays live apply progress. Prefer a server-side transactional apply endpoint and fallback to client-side recursion."

- Modal: Template Editor improvements [priority: high]
	- Purpose: Replace prompt()-based edits inside `admin/templates.php` with inline inputs inside the editor modal. Allow per-item edit/save/cancel, estimated minutes editing, and drag-reorder persistence.
	- Implementation notes:
		- Render each item with editable fields (text, estimated_minutes) and Save/Cancel controls.
		- Maintain a local dirty state and apply all changes on "Save Template" to reduce API chatter.
		- Add keyboard shortcuts (Enter to save, Esc to cancel) and ARIA labels.
	- Prompt for developer: "Implement per-item inline editors with ARIA attributes and validation (max length, numeric minutes). Persist on Save with a single sync operation (delete+recreate or diff-based)."

- Modal: Ticket selector modal [priority: medium]
	- Purpose: Reusable modal to pick a ticket (used by Apply Template and other cross-entity flows). Supports filtering by client, status, and a quick search.
	- Prompt for developer: "Build ticket selector modal that queries `/includes/crud.php?entity=tickets&q=...` and returns the selected id via a callback. Support keyboard selection and paging."

- Modal: Time entry / quick log [priority: medium]
	- Purpose: Lightweight modal to add or adjust `actual_minutes` on a ticket/item and optionally start/stop a timer for live logging.
	- Implementation notes:
		- Presets (15/30/60), small note field, start/stop timer UI, and a compact history briefly shown after logging.
	- Prompt for developer: "Create a lightweight time-entry modal that updates `todo_items.actual_minutes` (or posts time_log entries). Provide presets and a start/stop timer."

- Modal: Bulk actions modal [priority: low]
	- Purpose: Multi-select templates/items and perform batch operations: apply to multiple tickets, set estimate, delete, or export.
	- Prompt for developer: "Implement a bulk-actions modal that previews changes and applies them via batch API endpoints. Consider server-side batch endpoints for large sets."

- Modal: Ad-hoc checklist creation [priority: high]
	- Purpose: Allow admins to build a checklist on the fly (without using an existing template). The modal lets users add nested items, set per-item `estimated_minutes`, preview the resulting checklist summary, then either save it as a template or apply it immediately to one or more tickets.
	- Implementation notes:
		- Small modal UI that re-uses the template editor's nested-item builder controls (add, edit, reorder, add-subitem, delete) plus an `estimated_minutes` numeric input per item.
		- Preview pane: item count, total estimated minutes, and a small scrollable tree of items.
		- Actions: "Save as template" (POST to `entity=todo_templates` + items) and "Apply now" (either call existing `action=apply_template` with the newly-created template, or POST an `items` payload to a one-shot apply endpoint).
		- Keyboard-accessible and ARIA-labeled; show progress and result toast. Consider allowing multi-select apply in a future iteration.
	- Prompt for developer: "Create a reusable 'New List' modal that can be opened from Templates and Ticket pages. Support saving as template and applying immediately to a ticket. Validate minutes as integers and prevent empty lists."

- Accessibility & keyboard navigation pass [priority: high]
	- Purpose: Ensure all modals are accessible: aria-modal, focus-trap, keyboard nav, and screen-reader labels.
	- Prompt for developer: "Audit modal components and lists for keyboard focus and ARIA roles. Add focus traps and visible focus indicators."

- Server: apply-template transactional endpoint [priority: high]
	- Purpose: Server-side action that applies a template to a ticket in one DB transaction. Faster and atomic compared to client-side recursion.
	- Prompt for developer: "Add a new `action=apply_template` handler in `includes/crud.php` that accepts `template_id` and `ticket_id`, reads `todo_template_items` and creates `todo_items` in a transaction, returning a summary (created_count, errors)."

- UX: Toasts & progress indicators [priority: medium]
	- Purpose: Improve toasts with progress state for long operations (apply template), and show spinners/overlays inside modals.
	- Prompt for developer: "Create a toast/progress component that supports indeterminate/determinate progress, cancel actions, and logs. Use it during template apply and long saves."

Add these modal improvements to the project backlog and pick one to implement next (I can implement the Template Apply modal + server-side apply endpoint as a single task if you want the fastest, most robust UX improvement).
