<?php
// Secure session cookie flags (set before session_start)
$secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
session_set_cookie_params([
    'lifetime' => 0,
    'path' => '/',
    'domain' => '',
    'secure' => $secure,
    'httponly' => true,
    'samesite' => 'Lax',
]);

session_start();
// Basic security headers
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('Referrer-Policy: same-origin');

require_once '../includes/auth.php';
require_once '../includes/db.php';
require_once '../includes/csrf.php';

$errors = [];
$notices = [];

$action = $_POST['action'] ?? '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $token = $_POST['csrf_token'] ?? '';
    if (!csrf_verify($token)) {
        $errors[] = 'Invalid request. Please refresh and try again.';
    } else {
        if ($action === 'create') {
            $email = trim((string)($_POST['email'] ?? ''));
            $expiryPreset = (string)($_POST['expiry'] ?? '24h');
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $errors[] = 'Please provide a valid email address.';
            }

            // Check if already an admin
            if (!$errors) {
                $stmt = $pdo->prepare('SELECT id FROM admins WHERE email = ?');
                $stmt->execute([$email]);
                if ($stmt->fetch()) {
                    $errors[] = 'An account with this email already exists.';
                }
            }

            // Compute expiry
            $ttl = 24 * 3600; // default 24h
            if ($expiryPreset === '1h') $ttl = 3600;
            elseif ($expiryPreset === '24h') $ttl = 24 * 3600;
            elseif ($expiryPreset === '7d') $ttl = 7 * 24 * 3600;
            $expiresAt = (new DateTimeImmutable('@' . (time() + $ttl)))->setTimezone(new DateTimeZone(date_default_timezone_get()))->format('Y-m-d H:i:s');

            if (!$errors) {
                try {
                    $tokenVal = bin2hex(random_bytes(32));
                    $creatorId = isset($_SESSION['admin_id']) ? (int)$_SESSION['admin_id'] : null;
                    $stmt = $pdo->prepare('INSERT INTO admin_invites (token, email, expires_at, created_by_admin_id) VALUES (?, ?, ?, ?)');
                    $stmt->execute([$tokenVal, $email, $expiresAt, $creatorId]);
                    $notices[] = 'Invite created successfully.';
                    // Pass back the new token for UI copy
                    $_SESSION['last_invite_token'] = $tokenVal;
                    $_SESSION['last_invite_email'] = $email;
          // Attempt to send invite email
          try {
            require_once __DIR__ . '/../includes/mailer.php';
            $inviteLink = $baseUrl . $registerBase . $tokenVal;
            list($ok, $info) = send_invite_email($email, $inviteLink, ['from_name' => 'Checklist Admin', 'from_email' => 'no-reply@' . $_SERVER['HTTP_HOST']]);
            if ($ok) {
              $notices[] = 'Invite email sent.';
            } else {
              $errors[] = 'Invite created but email failed to send: ' . htmlspecialchars($info);
            }
          } catch (Throwable $e) {
            $errors[] = 'Invite created but email send failed.';
          }
                } catch (Throwable $e) {
                    $errors[] = 'Unable to create invite. Please try again.';
                }
            }
        } elseif ($action === 'revoke') {
            $id = (int)($_POST['id'] ?? 0);
            if ($id <= 0) {
                $errors[] = 'Invalid invite.';
            } else {
                try {
                    $stmt = $pdo->prepare('UPDATE admin_invites SET redeemed_at = NOW() WHERE id = ? AND redeemed_at IS NULL');
                    $stmt->execute([$id]);
                    if ($stmt->rowCount() > 0) {
                        $notices[] = 'Invite revoked.';
                    } else {
                        $errors[] = 'Invite not found or already used.';
                    }
                } catch (Throwable $e) {
                    $errors[] = 'Unable to revoke invite.';
                }
            }
        }
    }
}

// Load pending invites
$invites = [];
try {
    $q = 'SELECT id, email, token, expires_at, created_at FROM admin_invites WHERE redeemed_at IS NULL AND expires_at > NOW() ORDER BY created_at DESC';
    $stmt = $pdo->query($q);
    $invites = $stmt->fetchAll();
} catch (Throwable $e) {
    $errors[] = 'Unable to load invites.';
}

$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'];
$registerBase = rtrim(dirname($_SERVER['REQUEST_URI']), '/') . '/register.php?token=';
$lastToken = $_SESSION['last_invite_token'] ?? '';
$lastEmail = $_SESSION['last_invite_email'] ?? '';
unset($_SESSION['last_invite_token'], $_SESSION['last_invite_email']);
?>
<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>Create Admin Invite</title>
  <style>
    body { font-family: Arial, sans-serif; margin: 2em; }
    .container { max-width: 840px; margin: 0 auto; }
    .card { border: 1px solid #ddd; padding: 16px; margin-bottom: 16px; border-radius: 6px; }
    label { display:block; margin: 8px 0 4px; }
    input, select { width: 100%; padding: 8px; }
    button { padding: 8px 12px; margin-top: 8px; }
    table { border-collapse: collapse; width: 100%; }
    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
    th { background: #f4f4f4; }
    #toastContainer { position: fixed; top: 16px; right: 16px; z-index: 1000; display: flex; gap: 8px; flex-direction: column; }
    .toast { background: #333; color: #fff; padding: 10px 14px; border-radius: 4px; opacity: .95; }
    .toast.success { background: #2e7d32; }
    .toast.error { background: #c62828; }
    .toast.info { background: #1565c0; }
  </style>
</head>
<body>
<div class="container">
  <h1>Create Admin Invite</h1>
  <p><a href="dashboard.php">Back to Dashboard</a></p>

  <div id="toastContainer"></div>
  <?php foreach ($notices as $n): ?>
    <div class="toast success"><?= htmlspecialchars($n) ?></div>
  <?php endforeach; ?>
  <?php foreach ($errors as $e): ?>
    <div class="toast error"><?= htmlspecialchars($e) ?></div>
  <?php endforeach; ?>

  <div class="card">
    <h3>New Invite</h3>
    <form method="post" autocomplete="off">
      <input type="hidden" name="csrf_token" value="<?= htmlspecialchars(csrf_token()) ?>">
      <input type="hidden" name="action" value="create">
      <label for="email">Email</label>
      <input id="email" name="email" type="email" required value="<?= htmlspecialchars($lastEmail) ?>">

      <label for="expiry">Expiry</label>
      <select id="expiry" name="expiry">
        <option value="1h">1 hour</option>
        <option value="24h" selected>24 hours</option>
        <option value="7d">7 days</option>
      </select>

      <button type="submit">Create Invite</button>
    </form>

    <?php if ($lastToken): ?>
      <div style="margin-top:12px;">
        <label>Invite Link</label>
        <input id="inviteLink" type="text" readonly value="<?= htmlspecialchars($baseUrl . $registerBase . $lastToken) ?>">
        <button type="button" onclick="copyInvite()">Copy</button>
        <p style="font-size: 12px; color: #555;">Share this link over a secure channel only.</p>
      </div>
    <?php endif; ?>
  </div>

  <div class="card">
    <h3>Pending Invites</h3>
    <table>
      <thead>
        <tr>
          <th>ID</th>
          <th>Email</th>
          <th>Expires</th>
          <th>Link</th>
          <th>Actions</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($invites as $inv): ?>
        <tr>
          <td><?= htmlspecialchars($inv['id']) ?></td>
          <td><?= htmlspecialchars($inv['email']) ?></td>
          <td><?= htmlspecialchars($inv['expires_at']) ?></td>
          <td style="max-width:320px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap;">
              <?= htmlspecialchars($baseUrl . $registerBase . $inv['token']) ?>
          </td>
          <td>
            <button type="button" onclick="copyText('<?= htmlspecialchars($baseUrl . $registerBase . $inv['token']) ?>')">Copy</button>
            <form method="post" style="display:inline;" onsubmit="return confirm('Revoke this invite?')">
              <input type="hidden" name="csrf_token" value="<?= htmlspecialchars(csrf_token()) ?>">
              <input type="hidden" name="action" value="revoke">
              <input type="hidden" name="id" value="<?= htmlspecialchars($inv['id']) ?>">
              <button type="submit">Revoke</button>
            </form>
          </td>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>
  </div>
</div>
<script>
function showToast(msg, type) {
  var c = document.getElementById('toastContainer');
  var t = document.createElement('div');
  t.className = 'toast ' + (type||'info');
  t.textContent = msg;
  c.appendChild(t);
  setTimeout(function(){ t.remove(); }, 3000);
}

function copyInvite(){
  var el = document.getElementById('inviteLink');
  if (!el) return;
  el.select();
  el.setSelectionRange(0, 99999);
  navigator.clipboard.writeText(el.value).then(function(){
    showToast('Copied invite link', 'success');
  }).catch(function(){
    showToast('Unable to copy', 'error');
  });
}
function copyText(text){
  navigator.clipboard.writeText(text).then(function(){
    showToast('Copied', 'success');
  }).catch(function(){
    showToast('Unable to copy', 'error');
  });
}
</script>
</body>
</html>
